/** @file   spritetools.h
 * @brief   Declaration of SpriteTools class.
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/01/29 20:18:38 $
 * @author  Tomi Lamminsaari
 */


#ifndef H_TPL_SPRITETOOLS_H
#define H_TPL_SPRITETOOLS_H

#include <vector>
#include <string>
#include <allegro.h>
#include "eng2d_dll.h"
#include "datatypes.h"


namespace eng2d {


/** @class  SpriteTools
 * @brief   Provides some utilities for sprite management.
 * @author  Tomi Lamminsaari
 *
 * All the members are static.
 */
class DLLIMPORT SpriteTools
{
public:

  ///
  /// Static members and methods
  /// ==========================
  
  /** Rotates the bitmap 'pSrc' in 'frameCount' number of stages a around
   * itself and saves each stage to a vector as independent bitmap.
   * @param     pSrc              Pointer to source bitmap.
   * @param     frameCount        Number of stages the rotation has.
   * @return    A vector that has pointers to the rotated bitmaps.
   */
  static std::vector<BITMAP*> getRotatedSprites( BITMAP* pSrc, int frameCount );
  
  /** As 'getRotatedSprites(...)'-method but results rle-sprites.
   * @param     pSrc              Pointer to source bitmap.
   * @param     frameCount        Number of stages the rotation has.
   * @return    A vector that has pointers to the rotated rle-sprites.
   */
  static std::vector<RLE_SPRITE*> getRotatedRLE( BITMAP* pSrc, int frameCount );

  /** Destroys the bitmaps from given vector.
   * @param     bmtable           The vector that has pointers to the bitmaps
   *                              that should be deleted.
   */
  static void flushBitmaps( std::vector<BITMAP*>& bmtable );
  
  /** Destroyes the rle-sprites from given vector.
   * @param     bmtable           The vector that has pointer to the
   *                              rle-sprites that should be destroyed.
   */
  static void flushRLE( std::vector<RLE_SPRITE*>& bmtable );

  /** Loads the bitmap from file and converts it to a rle-sprite.
   * @param     filename          Name of the bitmapfile.
   * @return    Pointer to loaded rle-sprite or null-pointer if loading failed.
   */
  static RLE_SPRITE* load_RLE_Sprite( const std::string& filename );

  /** Loads a set bitmaps and stores them to given vector.
   * @param     filename          Name of the filelist-file. This file contains
   *                              the bitmapfiles to be loaded.
   * @param     pT                Pointer to vector where we store the
   *                              loaded bitmaps.
   * @return    Nonzero if we fail to load some of the bitmaps.
   */
  static int loadBitmapSet( const std::string& filename,
                            std::vector<BITMAP*>* pT );
                            
  /** Loads a set of rle-sprites. The loaded images will be stored in given
   * vector.
   * @param     filename          Name of the filelist-fie. This file contains
   *                              the bitmapfiles to be loaded.
   * @param     pT                Pointer to vector where we store the loaded
   *                              rle-sprites.
   * @return    Nonzero if we fail to load some of the bitmaps.
   */
  static int loadRleSpriteSet( const std::string& filename,
                               std::vector<RLE_SPRITE*>* pT );
                               
  /** Creates a copy of given bitmap and converts it to given format. Only
   * bitmaps and rle-sprites are supported.
   * @param   aSource           Pointer to source allegro bitmap.
   * @param   aConvertTo        Content type where to convert.
   * @return  Pointer to a copy graphic. If fails, a nullpointer is returned.
   *          You must typecast this pointer to correct format. The ownership
   *          is transferred.
   */
  static void* ConvertToGraphicFormat( const BITMAP* aSource,
                                       GfxContentType aConvertTo );

  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructor.
   */
  SpriteTools();
  
  /** Copy constructor.
   */
  SpriteTools(const SpriteTools& rO);
  
  /** Destructor.
   */
  ~SpriteTools();
  
  /** Operator =
   */
  SpriteTools& operator = (const SpriteTools& rO);

};

};

#endif
